<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/api_helpers.php';

$activePage = 'cleared_credit_sales';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get currency from store settings
$stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
$stmt->execute();
$currency = $stmt->fetchColumn() ?: 'Ksh';

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'website')");
$stmt->execute();
$store_settings = [];
while ($row = $stmt->fetch()) {
    $store_settings[$row['setting_key']] = $row['setting_value'];
}

$storeName = $store_settings['store_name'] ?? 'Store Name';
$storeAddress = $store_settings['store_address'] ?? 'Store Address';
$storePhone = $store_settings['store_phone'] ?? 'Store Phone';
$storeEmail = $store_settings['store_email'] ?? 'Store Email';
$storeWebsite = $store_settings['website'] ?? 'Store Website';

// Retrieve store logo data
$logoData = null;
$logoName = '';
$stmt = $pdo->prepare("SELECT setting_value, setting_binary FROM store_settings WHERE setting_key = 'store_logo' AND setting_group = 'store_info'");
$stmt->execute();
$logo = $stmt->fetch();
if ($logo && $logo['setting_binary']) {
    $logoData = $logo['setting_binary'];
    $logoName = $logo['setting_value'];
}

// --- Fetch all credit transactions for grouping cleared credits ---
$all_credits_query = "SELECT ct.*, s.invoice_number, s.created_at as sale_date, 
    c.name as customer_name,
    GROUP_CONCAT(p.name SEPARATOR ', ') as items,
    GROUP_CONCAT(CONCAT(p.name, ' (', si.quantity, ' x ', si.price, ')') SEPARATOR ', ') as items_with_prices,
    COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as amount_paid,
    COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as amount_waived,
    CASE 
        WHEN EXISTS (SELECT 1 FROM credit_payments WHERE credit_transaction_id = ct.id) THEN 'paid'
        ELSE 'pending'
    END as status
    FROM credit_transactions ct
    JOIN sales s ON ct.sale_id = s.id
    JOIN customers c ON ct.customer_id = c.id
    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
    LEFT JOIN sale_items si ON s.id = si.sale_id
    LEFT JOIN products p ON si.product_id = p.id
    GROUP BY ct.id ORDER BY s.created_at DESC";
$all_credits_stmt = $pdo->prepare($all_credits_query);
$all_credits_stmt->execute();
$all_credits = $all_credits_stmt->fetchAll();

// Group all credits by customer for cleared credits
$all_credits_by_customer = [];
$cleared_credits_by_customer = [];
foreach ($all_credits as $credit) {
    $cid = $credit['customer_id'];
    if (!isset($all_credits_by_customer[$cid])) {
        $all_credits_by_customer[$cid] = [
            'customer_name' => $credit['customer_name'] ?? $credit['name'] ?? 'Unknown',
            'total_credit' => 0,
            'total_paid' => 0,
            'total_waived' => 0,
            'total_balance' => 0,
            'transactions' => []
        ];
    }
    $paid = $credit['amount_paid'];
    // Use waived amount from the main query
    $waived = $credit['amount_waived'];
    $total_paid_and_waived = $paid + $waived;
    // Only include fully paid invoices
    if (abs($credit['amount'] - $total_paid_and_waived) > 0.01) continue;
    $all_credits_by_customer[$cid]['total_credit'] += $credit['amount'];
    $all_credits_by_customer[$cid]['total_paid'] += $paid;
    $all_credits_by_customer[$cid]['total_waived'] += $waived;
    $all_credits_by_customer[$cid]['total_balance'] += $credit['amount'] - ($paid + $waived);
    $credit['amount_waived'] = $waived;
    $all_credits_by_customer[$cid]['transactions'][] = $credit;
}
// Separate cleared credits (total_balance == 0)
foreach ($all_credits_by_customer as $cid => $group) {
    if ($group['total_credit'] > 0 && abs($group['total_balance']) < 0.01) {
        $cleared_credits_by_customer[$cid] = $group;
    }
}

// Calculate Amount Paid and Top 3 Best Paying Customers
$amountPaid = 0;
$topCustomers = [];
foreach ($cleared_credits_by_customer as $cid => $group) {
    $amountPaid += $group['total_paid'];
    $topCustomers[] = [
        'customer_name' => $group['customer_name'],
        'total_paid' => $group['total_paid']
    ];
}
// Sort top customers by total_paid descending
usort($topCustomers, function($a, $b) { return $b['total_paid'] <=> $a['total_paid']; });
$topCustomers = array_slice($topCustomers, 0, 3);

// Calculate Top 3 Best Paying Months
$stmt = $pdo->prepare("SELECT DATE_FORMAT(created_at, '%M %Y') as month, SUM(amount) as total_paid FROM credit_payments WHERE amount > 0 GROUP BY month ORDER BY total_paid DESC LIMIT 3");
$stmt->execute();
$topMonths = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cleared Credit Sales - POS System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        :root {
            --primary-gradient: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
            --success-gradient: linear-gradient(135deg, #10b981 0%, #059669 100%);
            --warning-gradient: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
            --info-gradient: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%);
        }
        
        body { 
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            font-family: 'Inter', sans-serif;
            min-height: 100vh;
        }

        .card { 
            border-radius: 24px; 
            box-shadow: 
                0 4px 24px rgba(0,0,0,0.05),
                0 1px 2px rgba(0,0,0,0.1);
            border: none; 
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1); 
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
        }

        .card:hover { 
            box-shadow: 
                0 8px 32px rgba(0,0,0,0.08),
                0 2px 4px rgba(0,0,0,0.12);
            transform: translateY(-4px); 
        }

        .stat-card { 
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: 16px; 
            box-shadow: 
                0 4px 24px rgba(0,0,0,0.05),
                0 1px 2px rgba(0,0,0,0.1);
            margin-bottom: 0.25rem; 
            padding: 0.15rem 0.5rem;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(135deg, rgba(255,255,255,0.1) 0%, rgba(255,255,255,0) 100%);
            pointer-events: none;
        }

        .stat-card:hover {
            transform: translateY(-4px);
            box-shadow: 
                0 8px 32px rgba(0,0,0,0.08),
                0 2px 4px rgba(0,0,0,0.12);
        }

        .stat-card.success { 
            border-left: 4px solid #10b981;
            background: linear-gradient(135deg, rgba(16, 185, 129, 0.05) 0%, rgba(255, 255, 255, 0.95) 100%);
        }

        .stat-card.info { 
            border-left: 4px solid #3b82f6;
            background: linear-gradient(135deg, rgba(59, 130, 246, 0.05) 0%, rgba(255, 255, 255, 0.95) 100%);
        }

        .stat-card.primary { 
            border-left: 4px solid #06b6d4;
            background: linear-gradient(135deg, rgba(6, 182, 212, 0.05) 0%, rgba(255, 255, 255, 0.95) 100%);
        }

        .stat-card.warning { 
            border-left: 4px solid #f59e0b;
            background: linear-gradient(135deg, rgba(245, 158, 11, 0.05) 0%, rgba(255, 255, 255, 0.95) 100%);
        }

        .stat-label { 
            color: #64748b; 
            font-size: 0.65rem; 
            font-weight: 600; 
            text-transform: uppercase; 
            letter-spacing: 0.05em; 
            margin-bottom: 0.02rem;
            text-shadow: 0 1px 1px rgba(255,255,255,0.8);
        }

        .stat-value { 
            font-size: 0.85rem; 
            font-weight: 800; 
            color: #1e293b; 
            margin: 0.02rem 0;
            text-shadow: 0 1px 1px rgba(255,255,255,0.8);
        }

        .stat-card.success .stat-value,
        .stat-card.warning .stat-value {
            font-size: 1.8rem;
            background: var(--primary-gradient);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            text-shadow: none;
            line-height: 1.2;
        }

        .stat-card.success .stat-label,
        .stat-card.warning .stat-label {
            font-size: 0.85rem;
            color: #475569;
            text-shadow: none;
            margin-bottom: 0.1rem;
        }

        .modern-table { 
            border-radius: 24px;
            overflow: hidden;
            box-shadow: 
                0 4px 24px rgba(0,0,0,0.05),
                0 1px 2px rgba(0,0,0,0.1);
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
        }

        .modern-table th { 
            background: rgba(248, 250, 252, 0.95);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            font-weight: 700; 
            color: #334155; 
            border-bottom: 2px solid rgba(226, 232, 240, 0.8);
            position: sticky; 
            top: 0; 
            z-index: 2;
            padding: 0.35rem 0.5rem;
            font-size: 0.75rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            text-shadow: 0 1px 1px rgba(255,255,255,0.8);
            line-height: 1.2;
        }

        .modern-table td { 
            vertical-align: middle; 
            color: #334155;
            padding: 0.35rem 0.5rem;
            border-bottom: 1px solid rgba(226, 232, 240, 0.8);
            font-size: 0.8rem;
            transition: all 0.3s ease;
            line-height: 1.2;
        }

        .modern-table tr.cleared-customer-row:hover { 
            background: rgba(241, 245, 249, 0.8);
            cursor: pointer; 
            transition: all 0.3s ease;
        }

        .modern-table tr.cleared-transaction-row { 
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            background: rgba(248, 250, 252, 0.5);
        }

        .modern-table tr.cleared-transaction-row td {
            padding: 0.25rem 0.5rem;
        }

        .modern-table .table-sm th,
        .modern-table .table-sm td {
            padding: 0.25rem 0.5rem;
            font-size: 0.75rem;
            line-height: 1.2;
        }

        .modern-table .badge {
            padding: 0.2em 0.5em;
            font-size: 0.7rem;
            line-height: 1.2;
        }

        .modern-table .btn-sm {
            padding: 0.2rem 0.5rem;
            font-size: 0.7rem;
            line-height: 1.2;
        }

        .badge {
            padding: 0.35em 0.75em;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.75rem;
            letter-spacing: 0.02em;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            transition: all 0.3s ease;
        }

        .badge.bg-info { 
            background: var(--primary-gradient) !important;
            color: #fff;
        }

        .badge.bg-success { 
            background: var(--success-gradient) !important;
            color: #fff;
        }

        .badge.bg-warning { 
            background: var(--warning-gradient) !important;
            color: #fff;
        }

        .btn {
            padding: 0.5rem 1rem;
            border-radius: 8px;
            font-weight: 600;
            letter-spacing: 0.02em;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
            font-size: 0.85rem;
        }

        .btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(135deg, rgba(255,255,255,0.1) 0%, rgba(255,255,255,0) 100%);
            pointer-events: none;
        }

        .btn-outline-primary {
            border: 2px solid #3b82f6;
            color: #3b82f6;
            background: transparent;
        }

        .btn-outline-primary:hover {
            background: var(--primary-gradient);
            border-color: transparent;
            color: #fff;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(59,130,246,0.2);
        }

        .btn-primary {
            background: var(--primary-gradient);
            border: none;
            color: #fff;
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(59,130,246,0.2);
        }

        .input-group {
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 
                0 4px 24px rgba(0,0,0,0.05),
                0 1px 2px rgba(0,0,0,0.1);
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
        }

        .input-group-text {
            border: none;
            background: transparent;
            color: #64748b;
            padding: 0.5rem 0.75rem;
            font-size: 0.85rem;
        }

        .form-control {
            border: none;
            padding: 0.5rem 0.75rem;
            font-size: 0.85rem;
            background: transparent;
        }

        .form-control:focus {
            box-shadow: none;
            background: rgba(255, 255, 255, 0.8);
        }

        .top-customers-list { 
            list-style-position: inside;
            padding-left: 0;
            margin-bottom: 0;
        }

        .top-customers-list li { 
            font-size: 0.85rem; 
            margin-bottom: 0.15rem;
            color: #475569;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.15rem;
            border-radius: 6px;
            transition: all 0.3s ease;
            line-height: 1.2;
        }

        .top-customers-list li:last-child {
            margin-bottom: 0;
        }

        .top-customers-list li b {
            color: #1e293b;
            font-weight: 700;
            margin-right: 0.25rem;
        }

        .top-customers-list li small {
            margin-left: 0.25rem;
        }

        .footer-modern { 
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            border-top: 1px solid rgba(229, 231, 235, 0.8);
            color: #64748b; 
            text-align: center; 
            padding: 1.5rem 0; 
            font-size: 0.95rem; 
            margin-top: 2rem; 
            border-radius: 0 0 24px 24px;
            box-shadow: 
                0 -4px 24px rgba(0,0,0,0.05),
                0 -1px 2px rgba(0,0,0,0.1);
        }

        @media (max-width: 768px) {
            .stat-value { font-size: 0.9rem; }
            .stat-label { font-size: 0.7rem; }
            .top-customers-list li { font-size: 0.8rem; }
            .modern-table th,
            .modern-table td {
                padding: 0.5rem;
                font-size: 0.8rem;
            }
            .btn {
                padding: 0.35rem 0.75rem;
                font-size: 0.8rem;
            }
        }
    </style>
</head>
<body>
<div class="container-fluid py-4">
    <div class="row">
        <?php include '../includes/sidebar.php'; ?>
        <?php include '../includes/topbar.php'; ?>
        <div class="col-md-10 ms-sm-auto px-md-4">
            <div class="container-fluid mt-4">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="mb-0 fw-bold">Cleared Credit Sales</h2>
                    <div class="d-flex align-items-center gap-3">
                        <div class="d-flex align-items-center gap-2">
                            <div class="input-group" style="width: 200px;">
                                <span class="input-group-text bg-white border-end-0"><i class="fas fa-calendar"></i></span>
                                <input type="date" class="form-control border-start-0" id="startDate" name="startDate">
                            </div>
                            <span class="text-muted">to</span>
                            <div class="input-group" style="width: 200px;">
                                <span class="input-group-text bg-white border-end-0"><i class="fas fa-calendar"></i></span>
                                <input type="date" class="form-control border-start-0" id="endDate" name="endDate">
                            </div>
                            <button class="btn btn-primary" id="filterDates">
                                <i class="fas fa-filter"></i> Filter
                            </button>
                        </div>
                    <a href="credit.php" class="btn btn-outline-primary"><i class="fas fa-arrow-left"></i> Back to Credit Transactions</a>
                    </div>
                </div>
                <div class="row mb-4">
                    <div class="col-md-3 mb-3 mb-md-0">
                        <div class="card stat-card success h-100">
                            <div class="card-body d-flex flex-column justify-content-center align-items-center text-center">
                                <div class="stat-label">Amount Paid</div>
                                <div class="stat-value"><?php echo $currency . ' ' . number_format($amountPaid, 2); ?></div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3 mb-md-0">
                        <div class="card stat-card warning h-100">
                            <div class="card-body d-flex flex-column justify-content-center align-items-center text-center">
                                <div class="stat-label">Amount Waived</div>
                                <div class="stat-value"><?php 
                                    $totalWaived = 0;
                                    foreach ($cleared_credits_by_customer as $group) {
                                        $totalWaived += $group['total_waived'];
                                    }
                                    echo $currency . ' ' . number_format($totalWaived, 2); 
                                ?></div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3 mb-md-0">
                        <div class="card stat-card info h-100">
                            <div class="card-body">
                                <div class="stat-label mb-2">Top 3 Best Paying Customers</div>
                                <ol class="top-customers-list" style="margin-bottom:0;">
                                    <?php
                                    foreach ($topCustomers as $i => $c):
                                        $trend = '';
                                        if ($i === 0) {
                                            $trend = '<small style="color:#64748b;">&mdash;</small>';
                                        } else if ($c['total_paid'] > $topCustomers[$i-1]['total_paid']) {
                                            $trend = '<small style="color:#10b981;"><i class="fas fa-arrow-up"></i></small>';
                                        } else if ($c['total_paid'] < $topCustomers[$i-1]['total_paid']) {
                                            $trend = '<small style="color:#ef4444;"><i class="fas fa-arrow-down"></i></small>';
                                        } else {
                                            $trend = '<small style="color:#64748b;">&mdash;</small>';
                                        }
                                    ?>
                                    <li><b><?php echo htmlspecialchars($c['customer_name']); ?></b> - <?php echo $currency . ' ' . number_format($c['total_paid'], 2); ?> <?php echo $trend; ?></li>
                                    <?php endforeach; ?>
                                </ol>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card stat-card primary h-100">
                            <div class="card-body">
                                <div class="stat-label mb-2">Top 3 Best Paying Months</div>
                                <ol class="top-customers-list" style="margin-bottom:0;">
                                    <?php
                                    foreach ($topMonths as $i => $m):
                                        $trend = '';
                                        if ($i === 0) {
                                            $trend = '<small style="color:#64748b;">&mdash;</small>';
                                        } else if ($m['total_paid'] > $topMonths[$i-1]['total_paid']) {
                                            $trend = '<small style="color:#10b981;"><i class="fas fa-arrow-up"></i></small>';
                                        } else if ($m['total_paid'] < $topMonths[$i-1]['total_paid']) {
                                            $trend = '<small style="color:#ef4444;"><i class="fas fa-arrow-down"></i></small>';
                                        } else {
                                            $trend = '<small style="color:#64748b;">&mdash;</small>';
                                        }
                                    ?>
                                    <li><b><?php echo htmlspecialchars($m['month']); ?></b> - <?php echo $currency . ' ' . number_format($m['total_paid'], 2); ?> <?php echo $trend; ?></li>
                                    <?php endforeach; ?>
                                </ol>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="card">
                    <div class="card-header bg-white d-flex justify-content-between align-items-center flex-wrap gap-2">
                        <h5 class="mb-0 fw-bold">Cleared Credit Sales</h5>
                        <input type="text" id="liveSearchInput" class="form-control" style="max-width:260px;" placeholder="Search customer...">
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table modern-table table-hover mb-0" id="clearedCreditsTable">
                                <thead>
                                    <tr>
                                        <th>Customer</th>
                                        <th>Total Credit</th>
                                        <th>Total Paid</th>
                                        <th>Total Waived</th>
                                        <th>Total Balance</th>
                                        <th>Credits</th>
                                        <th>Print</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($cleared_credits_by_customer as $cid => $group): ?>
                                    <tr class="cleared-customer-row" data-customer-id="<?php echo $cid; ?>">
                                        <td>
                                            <span class="toggle-cleared-transactions" data-customer-id="<?php echo $cid; ?>" style="text-decoration:underline; cursor:pointer; color:#2563eb;">
                                                <?php echo htmlspecialchars($group['customer_name']); ?>
                                                <i class="fas fa-chevron-down ms-2"></i>
                                            </span>
                                        </td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_credit'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_waived'], 2); ?></td>
                                        <td><?php echo $currency . ' ' . number_format($group['total_balance'], 2); ?></td>
                                        <td><span class="badge bg-info"><?php echo count($group['transactions']); ?> credits</span></td>
                                        <td><button class="btn btn-sm btn-outline-primary print-cleared-receipt" data-customer-id="<?php echo $cid; ?>"><i class="fas fa-print"></i> Print</button></td>
                                    </tr>
                                    <tr class="cleared-transaction-row" data-customer-id="<?php echo $cid; ?>" style="display:none; background:#f8fafc;">
                                        <td colspan="7">
                                            <div class="p-3">
                                                <h6 class="mb-3">Credit Items for <?php echo htmlspecialchars($group['customer_name']); ?></h6>
                                                <div class="table-responsive">
                                                    <table class="table table-sm">
                                                        <thead>
                                                            <tr>
                                                                <th>Date</th>
                                                                <th>Invoice</th>
                                                                <th>Items</th>
                                                                <th>Item Price</th>
                                                                <th>Amount</th>
                                                                <th>Waived</th>
                                                                <th>Paid</th>
                                                                <th>Status</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach ($group['transactions'] as $credit): ?>
                                                            <?php
                                                            // Use waived amount from the credit data
                                                            $waived = $credit['amount_waived'];
                                                            $paid = $credit['amount_paid'];
                                                            $status = '';
                                                            $rowStyle = '';
                                                            $statusBadge = '';
                                                            if ($waived > 0 && $paid < $credit['amount']) {
                                                                $status = 'Waived';
                                                                $rowStyle = 'background-color: #fffbe6;'; // yellow
                                                                $statusBadge = '<span class="badge bg-warning text-dark">Waived</span>';
                                                            } else {
                                                                $status = 'Fully Paid';
                                                                $rowStyle = '';
                                                                $statusBadge = '<span class="badge bg-success">Fully Paid</span>';
                                                            }
                                                            ?>
                                                            <tr style="<?php echo $rowStyle; ?>">
                                                                <td><?php echo date('M d, Y', strtotime($credit['sale_date'])); ?></td>
                                                                <td><?php echo htmlspecialchars($credit['invoice_number']); ?></td>
                                                                <td>
                                                                    <?php
                                                                    $items = $credit['items'] ?? '';
                                                                    if ($items) {
                                                                        $itemsArr = array_map('trim', explode(',', $items));
                                                                        $itemCounts = [];
                                                                        foreach ($itemsArr as $item) {
                                                                            if ($item === '') continue;
                                                                            if (!isset($itemCounts[$item])) {
                                                                                $itemCounts[$item] = 0;
                                                                            }
                                                                            $itemCounts[$item]++;
                                                                        }
                                                                        $itemStrings = [];
                                                                        foreach ($itemCounts as $name => $qty) {
                                                                            if ($qty > 1) {
                                                                                $itemStrings[] = htmlspecialchars($name) . ' (<b>*' . $qty . '</b>)';
                                                                            } else {
                                                                                $itemStrings[] = htmlspecialchars($name);
                                                                            }
                                                                        }
                                                                        echo implode(' + ', $itemStrings);
                                                                    } else {
                                                                        echo 'No items found';
                                                                    }
                                                                    ?>
                                                                </td>
                                                                <td>
                                                                    <?php
                                                                    $itemsWithPrices = $credit['items_with_prices'] ?? '';
                                                                    if ($itemsWithPrices) {
                                                                        $itemsArr = explode(',', $itemsWithPrices);
                                                                        $grouped = [];
                                                                        $total = 0;
                                                                        foreach ($itemsArr as $itemStr) {
                                                                            $itemStr = trim($itemStr);
                                                                            // Match pattern: Name (qty x price)
                                                                            if (preg_match('/^(.*?) \((\d+) x ([\d.]+)\)$/', $itemStr, $matches)) {
                                                                                $name = $matches[1];
                                                                                $qty = (float)$matches[2];
                                                                                $price = (float)$matches[3];
                                                                                $subtotal = $qty * $price;
                                                                                $total += $subtotal;
                                                                                $key = $name . '|' . $price;
                                                                                if (!isset($grouped[$key])) {
                                                                                    $grouped[$key] = ['name' => $name, 'qty' => 0, 'price' => $price];
                                                                                }
                                                                                $grouped[$key]['qty'] += $qty;
                                                                            } else {
                                                                                // If pattern doesn't match, treat as unique
                                                                                $grouped[$itemStr . '|other'] = ['name' => $itemStr, 'qty' => 1, 'price' => 0];
                                                                            }
                                                                        }
                                                                        $itemStrings = [];
                                                                        foreach ($grouped as $g) {
                                                                            if ($g['price'] > 0) {
                                                                                $itemStrings[] = htmlspecialchars("{$g['name']} ({$g['qty']} x {$g['price']})");
                                                                            } else {
                                                                                $itemStrings[] = htmlspecialchars($g['name']);
                                                                            }
                                                                        }
                                                                        echo implode(' + ', $itemStrings) . ' = <b>' . $currency . ' ' . number_format($total, 2) . '</b>';
                                                                    } else {
                                                                        echo 'No price info';
                                                                    }
                                                                    ?>
                                                                </td>
                                                                <td><b><?php echo $currency . ' ' . number_format($credit['amount'], 2); ?></b></td>
                                                                <td><?php echo $currency . ' ' . number_format($waived, 2); ?></td>
                                                                <td><?php echo $currency . ' ' . number_format($credit['amount_paid'], 2); ?></td>
                                                                <td><?php echo $statusBadge; ?></td>
                                                            </tr>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                    <!-- Hidden receipt template for this customer -->
                                    <template id="receipt-template-<?php echo $cid; ?>" data-customer-name="<?php echo htmlspecialchars($group['customer_name']); ?>" data-total-paid="<?php echo $group['total_paid']; ?>">
                                        <style>
                                            .receipt-watermark {
                                                position: fixed;
                                                top: 0; left: 0; width: 100vw; height: 100vh;
                                                display: flex; flex-direction: column; align-items: center; justify-content: center;
                                                opacity: 0.08;
                                                z-index: 0;
                                                text-align: center;
                                                pointer-events: none;
                                                user-select: none;
                                            }
                                            @media print {
                                                .receipt-watermark { position: fixed !important; }
                                            }
                                        </style>
                                        <div style="width:100%; max-width:210mm; font-family:monospace; font-size:13px; color:#222; margin:0 auto; position:relative;">
                                            <!-- Watermark -->
                                            <div class="receipt-watermark">
                                                <?php if ($logoData): ?>
                                                    <?php
                                                    $mimeType = 'image/jpeg';
                                                    if (strpos(strtolower($logoName), '.png') !== false) {
                                                        $mimeType = 'image/png';
                                                    } elseif (strpos(strtolower($logoName), '.gif') !== false) {
                                                        $mimeType = 'image/gif';
                                                    }
                                                    ?>
                                                    <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:120mm; max-height:120mm; display:block; margin:0 auto 8px auto;">
                                                <?php endif; ?>
                                                <div style="font-size:3rem; font-weight:900; color:#222; letter-spacing:2px; text-transform:uppercase;">
                                                    <?php echo htmlspecialchars($storeName); ?>
                                                </div>
                                                <div style="font-size:1.2rem; color:#222; margin-top:8px;">
                                                    <?php echo htmlspecialchars($storePhone); ?><br>
                                                    <?php echo htmlspecialchars($storeEmail); ?><br>
                                                    <?php echo htmlspecialchars($storeWebsite); ?>
                                                </div>
                                            </div>
                                            <div style="position:relative; z-index:1;">
                                                <div style="text-align:center; margin-bottom:8px;">
                                                    <div style="font-size:1.1rem; font-weight:900; color:#000; margin-bottom:2px;">CLEARED CREDIT SALES REPORT</div>
                                                    <?php if ($logoData): ?>
                                                    <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" style="max-width:40mm; max-height:30mm; display:block; margin:0 auto 12px auto;" onerror="this.style.display='none'">
                                                <?php endif; ?>
                                                <div style="font-size:1.2rem; font-weight:700; color:#222;"><?php echo htmlspecialchars($storeName); ?></div>
                                                <div><?php echo htmlspecialchars($storeAddress); ?></div>
                                                <div>Phone: <?php echo htmlspecialchars($storePhone); ?></div>
                                                <div>Email: <?php echo htmlspecialchars($storeEmail); ?></div>
                                                <div>Web: <?php echo htmlspecialchars($storeWebsite); ?></div>
                                            </div>
                                            <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                            <div><b>Customer:</b> <?php echo htmlspecialchars($group['customer_name']); ?></div>
                                                <div><b>Customer Phone:</b> <?php echo htmlspecialchars($group['customer_phone'] ?? 'N/A'); ?></div>
                                            <div><b>Date:</b> <?php echo date('Y-m-d H:i:s'); ?></div>
                                            <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                                <div style="text-align:center; font-weight:bold; font-size:1.1rem; margin-bottom:2px;">Cleared Credit Items:</div>
                                                <div style="border-bottom:2px dotted #222; margin-bottom:8px;"></div>
                                                <table style="width:100%; font-size:12px; border-collapse:collapse; border:1px solid #e5e7eb; background:#fff;">
                                                <thead>
                                                        <tr style="background:#f3f4f6; color:#222; font-weight:bold;">
                                                            <th style="text-align:left; padding:6px 4px; border:1px solid #e5e7eb;">Date</th>
                                                            <th style="text-align:left; padding:6px 4px; border:1px solid #e5e7eb;">Invoice</th>
                                                            <th style="text-align:left; padding:6px 4px; border:1px solid #e5e7eb;">Items</th>
                                                            <th style="text-align:left; padding:6px 4px; border:1px solid #e5e7eb;">Item Price</th>
                                                            <th style="text-align:right; padding:6px 4px; border:1px solid #e5e7eb;">Amount</th>
                                                            <th style="text-align:right; padding:6px 4px; border:1px solid #e5e7eb;">Waived</th>
                                                            <th style="text-align:right; padding:6px 4px; border:1px solid #e5e7eb;">Paid</th>
                                                            <th style="text-align:center; padding:6px 4px; border:1px solid #e5e7eb;">Status</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                        <?php $rowAlt = false; foreach ($group['transactions'] as $item): ?>
                                                    <?php
                                                    $waived = 0;
                                                    $waiveStmt = $pdo->prepare("SELECT SUM(amount) FROM credit_payments WHERE credit_transaction_id = ? AND is_waived = 1");
                                                    $waiveStmt->execute([$item['id']]);
                                                    $waived = $waiveStmt->fetchColumn();
                                                    $status = '';
                                                        $statusBadge = '';
                                                    if ($waived > 0 && $item['amount_paid'] < $item['amount']) {
                                                        $status = 'Waived';
                                                            $statusBadge = '<span style=\'color:#b45309; font-weight:bold;\'>Waived</span>';
                                                    } else {
                                                        $status = 'Fully Paid';
                                                            $statusBadge = '<span style=\'color:#16a34a; font-weight:bold;\'>Fully Paid</span>';
                                                        }
                                                        $rowAlt = !$rowAlt;
                                                        ?>
                                                        <tr style="background:<?php echo $rowAlt ? '#f9fafb' : '#fff'; ?>;">
                                                            <td style="padding:5px 4px; border:1px solid #e5e7eb;"><?php echo date('Y-m-d', strtotime($item['sale_date'])); ?></td>
                                                            <td style="padding:5px 4px; border:1px solid #e5e7eb;"><?php echo htmlspecialchars($item['invoice_number']); ?></td>
                                                            <td style="font-size:11px; padding:5px 4px; border:1px solid #e5e7eb;">
                                                                <?php
                                                                $items = $item['items'] ?? '';
                                                                if ($items) {
                                                                    $itemsArr = array_map('trim', explode(',', $items));
                                                                    $itemCounts = [];
                                                                    foreach ($itemsArr as $itm) {
                                                                        if ($itm === '') continue;
                                                                        if (!isset($itemCounts[$itm])) {
                                                                            $itemCounts[$itm] = 0;
                                                                        }
                                                                        $itemCounts[$itm]++;
                                                                    }
                                                                    $itemStrings = [];
                                                                    foreach ($itemCounts as $name => $qty) {
                                                                        if ($qty > 1) {
                                                                            $itemStrings[] = htmlspecialchars($name) . ' (<b>*' . $qty . '</b>)';
                                                                        } else {
                                                                            $itemStrings[] = htmlspecialchars($name);
                                                                        }
                                                                    }
                                                                    echo implode(' + ', $itemStrings);
                                                                } else {
                                                                    echo '';
                                                                }
                                                                ?>
                                                        </td>
                                                            <td style="font-size:11px; padding:5px 4px; border:1px solid #e5e7eb;">
                                                                <?php
                                                                $itemsWithPrices = $item['items_with_prices'] ?? '';
                                                                if ($itemsWithPrices) {
                                                                    $itemsArr = explode(',', $itemsWithPrices);
                                                                    $grouped = [];
                                                                    $total = 0;
                                                                    foreach ($itemsArr as $itemStr) {
                                                                        $itemStr = trim($itemStr);
                                                                        if (preg_match('/^(.*?) \((\d+) x ([\d.]+)\)$/', $itemStr, $matches)) {
                                                                            $name = $matches[1];
                                                                            $qty = (float)$matches[2];
                                                                            $price = (float)$matches[3];
                                                                            $subtotal = $qty * $price;
                                                                            $total += $subtotal;
                                                                            $key = $name . '|' . $price;
                                                                            if (!isset($grouped[$key])) {
                                                                                $grouped[$key] = ['name' => $name, 'qty' => 0, 'price' => $price];
                                                                            }
                                                                            $grouped[$key]['qty'] += $qty;
                                                                        } else {
                                                                            $grouped[$itemStr . '|other'] = ['name' => $itemStr, 'qty' => 1, 'price' => 0];
                                                                        }
                                                                    }
                                                                    $itemStrings = [];
                                                                    foreach ($grouped as $g) {
                                                                        if ($g['price'] > 0) {
                                                                            $itemStrings[] = htmlspecialchars("{$g['name']} ({$g['qty']} x {$g['price']})");
                                                                        } else {
                                                                            $itemStrings[] = htmlspecialchars($g['name']);
                                                                        }
                                                                    }
                                                                    echo implode(' + ', $itemStrings) . ' = <b>' . $currency . ' ' . number_format($total, 2) . '</b>';
                                                                } else {
                                                                    echo 'No price info';
                                                                }
                                                                ?>
                                                        </td>
                                                            <td style="text-align:right; padding:5px 4px; border:1px solid #e5e7eb;"><b><?php echo $currency . ' ' . number_format($item['amount'], 2); ?></b></td>
                                                            <td style="text-align:right; color:#dc2626; padding:5px 4px; border:1px solid #e5e7eb;"><?php echo $waived > 0 ? ($currency . ' ' . number_format($waived, 2)) : '-'; ?></td>
                                                            <td style="text-align:right; padding:5px 4px; border:1px solid #e5e7eb;"><b><?php echo $currency . ' ' . number_format($item['amount_paid'], 2); ?></b></td>
                                                            <td style="text-align:center; padding:5px 4px; border:1px solid #e5e7eb;"><?php echo $statusBadge; ?></td>
                                                    </tr>
                                                    <?php endforeach; ?>
                                                </tbody>
                                            </table>
                                            <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                            <div><b>Total Paid:</b> <?php echo $currency . ' ' . number_format($group['total_paid'], 2); ?></div>
                                            <div style="margin:12px 0 4px 0; text-align:center;">Thanks for shopping with <?php echo htmlspecialchars($storeName); ?>!</div>
                                            <div style="text-align:center; margin:8px 0;">
                                                <img id="receipt-qr" style="width:120px; height:120px;" />
                                            </div>
                                            <div style="border-top:1px dashed #222; margin:8px 0;"></div>
                                            <div style="text-align:center; font-size:11px;">Designed and developed by Vicotechnologies.co.ke<br>+254702214793</div>
                                            </div>
                                        </div>
                                    </template>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Set default dates (last 30 days)
    const today = new Date();
    const thirtyDaysAgo = new Date(today);
    thirtyDaysAgo.setDate(today.getDate() - 30);
    
    document.getElementById('startDate').value = thirtyDaysAgo.toISOString().split('T')[0];
    document.getElementById('endDate').value = today.toISOString().split('T')[0];

    // Date filter functionality
    document.getElementById('filterDates').addEventListener('click', function() {
        const startDate = new Date(document.getElementById('startDate').value);
        const endDate = new Date(document.getElementById('endDate').value);
        
        // Add one day to end date to include the entire end date
        endDate.setDate(endDate.getDate() + 1);
        
        document.querySelectorAll('#clearedCreditsTable tbody tr.cleared-customer-row').forEach(function(row) {
            const detailsRow = document.querySelector('.cleared-transaction-row[data-customer-id="' + row.getAttribute('data-customer-id') + '"]');
            if (detailsRow) {
                const transactions = detailsRow.querySelectorAll('tbody tr');
                let hasVisibleTransactions = false;
                
                transactions.forEach(function(transaction) {
                    const dateCell = transaction.querySelector('td:first-child');
                    if (dateCell) {
                        const transactionDate = new Date(dateCell.textContent);
                        if (transactionDate >= startDate && transactionDate < endDate) {
                            transaction.style.display = '';
                            hasVisibleTransactions = true;
                        } else {
                            transaction.style.display = 'none';
                        }
                    }
                });
                
                // Show/hide customer row based on visible transactions
                row.style.display = hasVisibleTransactions ? '' : 'none';
            }
        });
    });

    document.querySelectorAll('.toggle-cleared-transactions').forEach(function(link) {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            var cid = this.getAttribute('data-customer-id');
            var row = document.querySelector('.cleared-transaction-row[data-customer-id="' + cid + '"]');
            var chevron = this.querySelector('i.fas');
            if (row) {
                if (row.style.display === 'none' || row.style.display === '') {
                    row.style.display = 'table-row';
                    if (chevron) { chevron.classList.remove('fa-chevron-down'); chevron.classList.add('fa-chevron-up'); }
                } else {
                    row.style.display = 'none';
                    if (chevron) { chevron.classList.remove('fa-chevron-up'); chevron.classList.add('fa-chevron-down'); }
                }
            }
        });
    });
    document.querySelectorAll('.print-cleared-receipt').forEach(function(btn) {
        btn.addEventListener('click', function() {
            var cid = this.getAttribute('data-customer-id');
            var template = document.getElementById('receipt-template-' + cid);
            if (template) {
                var win = window.open('', '', 'width=900,height=1200');
                win.document.write('<html><head><title>Receipt</title><style>body { font-family: "Sans Serif", sans-serif; font-weight: bold; color: #000; }</style></head><body style="margin:0;">');
                win.document.write(template.innerHTML);
                win.document.write('</body></html>');
                win.document.close();
                
                // Generate QR code data for cleared credits
                var qrData = JSON.stringify({
                    store: '<?php echo htmlspecialchars($storeName); ?>',
                    customer: template.getAttribute('data-customer-name'),
                    total_paid: template.getAttribute('data-total-paid'),
                    date: new Date().toLocaleDateString(),
                    time: new Date().toLocaleTimeString(),
                    status: 'Cleared'
                });
                
                // Generate QR code using QRServer
                var qrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=' + encodeURIComponent(qrData);
                var qrImg = win.document.getElementById('receipt-qr');
                if (qrImg) {
                    qrImg.src = qrUrl;
                    // Wait for QR code to load before printing
                    qrImg.onload = function() {
                        win.focus();
                        setTimeout(function() { 
                            win.print();
                            // Close window after printing
                            setTimeout(function() {
                                win.close();
                            }, 500);
                        }, 300);
                    };
                    // Fallback in case QR code fails to load
                    qrImg.onerror = function() {
                        win.focus();
                        setTimeout(function() { 
                            win.print();
                            // Close window after printing
                            setTimeout(function() {
                                win.close();
                            }, 500);
                        }, 300);
                    };
                }
            }
        });
    });
    // Live search for Cleared Credit Sales
    var searchInput = document.getElementById('liveSearchInput');
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            var filter = this.value.toLowerCase();
            document.querySelectorAll('#clearedCreditsTable tbody tr.cleared-customer-row').forEach(function(row) {
                var nameCell = row.querySelector('td');
                var name = nameCell ? nameCell.textContent.toLowerCase() : '';
                row.style.display = name.includes(filter) ? '' : 'none';
                // Also hide the details row if the customer row is hidden
                var cid = row.getAttribute('data-customer-id');
                var detailsRow = document.querySelector('.cleared-transaction-row[data-customer-id="' + cid + '"]');
                if (detailsRow && row.style.display === 'none') {
                    detailsRow.style.display = 'none';
                }
            });
        });
    }
});
</script>
<?php include '../includes/footer.php'; ?>
</body>
</html> 